///////////////////////////////////////////
//
// RecMgrDBUtils.java
//
package Alkindi.Services.ServicesImpl;

import java.util.*;
import java.sql.*;
import java.math.*;
import oracle.jdbc.driver.*;
import Alkindi.Services.*;
import Alkindi.Data.*;
import Alkindi.Services.Util.*;
import Alkindi.Services.InternalData.*;

/* 
$Header: RecMgrDBUtils.java, 37, 5/10/01 12:04:06 PM, Schwartz, Joe$
$Log: 
 37   Alkindi Development1.36        5/10/01 12:04:06 PM  Schwartz, Joe  
      Renamed method getAlkindexInfo to getUserUCSubgroupStats and made it use
      InternalData class UserUCSubgroupData.
 36   Alkindi Development1.35        5/3/01 10:11:59 AM   Schwartz, Joe  
      Removed utility class. Modified insertRecs to use oridnary recList, since
      new Recommedation class stores scoring function and product cluster.
 35   Alkindi Development1.34        4/26/01 2:37:17 PM   Schwartz, Joe  
      Modifed to account for new InternalData package.
 34   Alkindi Development1.33        4/26/01 1:11:15 PM   Schwartz, Joe   Moved
      into new ServicesImpl package.
 33   Alkindi Development1.32        4/26/01 11:18:30 AM  Schwartz, Joe  
      Changed to account for connection being set to autocommit; removed several
      calls to con.commit().
 32   Alkindi Development1.31        3/2/01 4:59:27 PM    Schwartz, Joe  
      Modified for separate database search for Primary products.
 31   Alkindi Development1.30        2/27/01 1:01:56 PM   Schwartz, Joe  
      Working on optimizing stateful bean.
 30   Alkindi Development1.29        2/23/01 7:25:16 PM   Schwartz, Joe  
      Working on cache issues.
 29   Alkindi Development1.28        2/23/01 4:32:42 PM   Schwartz, Joe  
      Changed getPredictedRating to use new SP and return a float. Added state
      caching of PCWeights when all PCs are requested. Numerous small fixes. 
 28   Alkindi Development1.27        2/22/01 3:18:05 PM   Schwartz, Joe  
      Several changes to support new spec. Made stateful bean.
 27   Alkindi Development1.26        2/19/01 6:16:31 PM   Schwartz, Joe  
      Interin checkin: working on new RecMgr spec.
 26   Alkindi Development1.25        2/16/01 5:51:44 PM   Schwartz, Joe  
      Interim save: working off of new RecMgr spec. Adapting to use new objects
      from database. 
 25   Alkindi Development1.24        2/13/01 5:55:31 PM   Schwartz, Joe  
      Changed to account for new Product id int type and SparseRatingsArray.
 24   Alkindi Development1.23        2/5/01 3:08:23 PM    Schwartz, Joe  
      Changed to accomodate new AlkConn members.
 23   Alkindi Development1.22        1/25/01 9:58:07 PM   Schwartz, Joe  
      Working on speed, sensibility.
 22   Alkindi Development1.21        12/28/00 1:10:30 PM  Schwartz, Joe   Added
      Version Control header info.
      Changed package-scope members to protected.
 21   Alkindi Development1.20        12/26/00 6:57:16 PM  Schwartz, Joe   
 20   Alkindi Development1.19        12/26/00 5:37:00 PM  Schwartz, Joe   
 19   Alkindi Development1.18        12/20/00 5:44:57 PM  Schwartz, Joe   
 18   Alkindi Development1.17        12/19/00 3:51:58 PM  Schwartz, Joe   
 17   Alkindi Development1.16        12/18/00 6:08:24 PM  Schwartz, Joe   
 16   Alkindi Development1.15        12/18/00 4:35:38 PM  Schwartz, Joe   
 15   Alkindi Development1.14        12/18/00 12:05:39 PM Schwartz, Joe   Moved
      internal data classes to Utils package & regenerated classes from Rose.
 14   Alkindi Development1.13        12/18/00 11:25:22 AM Schwartz, Joe   
 13   Alkindi Development1.12        12/18/00 11:25:16 AM Schwartz, Joe   
 12   Alkindi Development1.11        12/6/00 7:11:58 PM   Schwartz, Joe   Added
      catch blocks for generic exceptions and finally blocks to close all
      database connections.
 11   Alkindi Development1.10        12/3/00 11:10:11 PM  Schwartz, Joe   Added
      new function getProductCountByPC to call new stored proc
      pkg_ALKINDI_RECOMMENDATION.sp_SEL_PC_NI.
 10   Alkindi Development1.9         12/3/00 10:40:17 PM  Schwartz, Joe   
 9    Alkindi Development1.8         12/3/00 7:51:44 PM   Schwartz, Joe  
      Changed getPCDataByUser as per Eugene's request.
 8    Alkindi Development1.7         11/29/00 2:06:34 PM  Schwartz, Joe  
      Changed to use stored procs for insert/update operations.
 7    Alkindi Development1.6         11/14/00 12:53:15 PM Schwartz, Joe   
 6    Alkindi Development1.5         11/7/00 2:04:34 PM   Schwartz, Joe  
      Modified to close DB connections.
 5    Alkindi Development1.4         11/6/00 7:05:59 PM   Schwartz, Joe   
 4    Alkindi Development1.3         11/6/00 11:50:21 AM  Schwartz, Joe   
 3    Alkindi Development1.2         10/22/00 10:38:46 AM Schwartz, Joe  
      AppianDelivery 10.20.00
 2    Alkindi Development1.1         10/22/00 10:34:10 AM Schwartz, Joe   
 1    Alkindi Development1.0         10/17/00 2:23:18 PM  Schwartz, Joe   
$
$NoKeywords$
 */

/**
 */
public class RecMgrDBUtils 
{
	
	/**
	 * Numnber of Scoring Functions
	 */
	protected static int propNumSF;
	
	/**
	 * Average Rating a Product must have to be recommended
	 */
	protected static int propB6;
	
	/**
	 * Constant term in exponential used to calculate the predicted rating.
	 */
	protected static double propAPrime;
	
	/**
	 * Default value for a predicted rating; used when the database has no stats for the product yet.
	 */
	protected static float propDefaultPR;
	static final String cName = "RecMgrDB";
	
	/**
	 * 	@param uid the user ID of the user in question
	 * 	@return void	
	 * 	@throws AlkExcept
	 * @roseuid 3A4B6610009C
	 */
	public void flushUserRecs(long uid) throws AlkExcept 
	{
		Connection con = null;
		
		try {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_RECOMMENDATION.SP_FLUSH_RECOMMENDATION_TABLE (?,?,?)}");
			cs.setLong(1, uid);
			cs.setDate(2, new java.sql.Date(System.currentTimeMillis()));
			cs.registerOutParameter(3, Types.INTEGER);

			cs.execute();
			int error = cs.getInt(3);
			if (error != 0) {
					throw new AlkExcept("", error);
			}
			cs.close();
		}
		catch (SQLException E){
				throw new AlkExcept("SQL Exception in SP_FLUSH_RECOMMENDATION_TABLE", 6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2510);
		}
		finally {
			try { con.close(); } catch (Exception e) {}
		}
	}
	
	/**
	 * Retrieves data used to calculate the given user's Alkindex.  Returns an ArrayList of UserUCSubgroupData objects.
	 * @param user the user in question
	 * @return ArrayList
	 * @throws AlkExcept
	 * @see UserUCSubgroupData
	 * @roseuid 3A4B6610008C
	 */
	public ArrayList getUserUCSubgroupStats(SystemUser user) throws AlkExcept 
	{
		ArrayList retList = new ArrayList();
		Connection con = null;
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = 
				con.prepareCall("{ call pkg_ALKINDI_RECOMMENDATION.SP_SEL_UC_SUBGROUP_BY_USER(?,?,?)}");  
			cs.setLong (1, user.id);
			cs.registerOutParameter(2, OracleTypes.CURSOR );
			cs.registerOutParameter(3, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(3);
			if (error != 0) {
					throw new AlkExcept("", error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(2); 
			while (rs.next()) 
			{
				UserUCSubgroupData uusd = new UserUCSubgroupData();
				uusd.pcID  = rs.getInt(1);
				uusd.ni = rs.getInt(2);
				uusd.fi = rs.getDouble(3) / uusd.ni;
				uusd.BBi = rs.getDouble(4);
				uusd.Bi = rs.getDouble(5);
				retList.add(uusd);
			}
			rs.close();
			cs.close();
			
		} //END try

		catch(SQLException se) 
		{
			throw new AlkExcept("SQL Exception in RecMgrDBUtils.getAlkindexInfo. Message: "  + 
				se.getMessage(),6004);
		} 
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2509);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
		return retList;
	}
	
	/**
	 * 	Retrieves statistics used to calculate the average rating for a given Product.
	 * 	@param prodID the Product ID of the Product in question
	 * 	@return double
	 * 	@throws AlkExcept
	 * @roseuid 3A4B6610006D
	 */
	public double getAverageRating(long prodID) throws AlkExcept 
	{
		Connection con = null;
		double averageRating = 0;
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = 
				con.prepareCall("{ call pkg_ALKINDI_RECOMMENDATION.sp_SEL_AVG_EVAL_BY_PROD(?,?)}");  
			cs.setLong (1, prodID);
			cs.registerOutParameter(2, Types.DOUBLE );
			cs.executeQuery();
			averageRating = cs.getDouble(2); 
			cs.close();
		} //END try

		catch(SQLException se) 
		{
			throw new AlkExcept("SQL Exception in RecMgrDBUtils.getAverageRating." +
				"\nStored Procedure is sp_SEL_AVG_EVAL_BY_PROD." + 
				"\nProduct is " + prodID + " Message: " +
				se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2507);
		}
		finally {
			try {	con.close(); } catch(Exception e) {}
		}
		return averageRating;
	}
	
	/**
	 * Retrieves PC statistics for a given user. These are used to calculate PC Weights. 
	 * <br>A <tt>UserEvalData</tt> object reference is to be passed to this method; it is set to one retrieved from the database. This is included here because that data is also required to calculate PC Weights.
	 * @param user The user for whom to get the ProductCluster stats.
	 * @param evalData A UserEvalData object reference, used to return overall statistics about the user's evaluations. 
	 * @return A HashMap of PCRecData objects.
	 * @throws AlkExcept
	 * @roseuid 3A4B6610002F
	 */
	public HashMap getPCData(SystemUser user, UserEvalData evalData) throws AlkExcept 
	{
	  
		Connection con = null;
		HashMap retMap = new HashMap();
		String mName = "getPCData";
		try
         {
			con = AlkConn.getConnection();
			
			//	Clear the UserEvalData reference passed in to this function.
			//
         	evalData = null;
         	
         	//	Map database object types to Java objects for this connection.
			//
			Map typeMap = con.getTypeMap();
			typeMap.put(UserEvalData.DEFAULT_SQL_TYPE, UserEvalData.class);
			//	Temporarily remove other object types from connection map
			//
			//typeMap.put(PCRecData.DEFAULT_SQL_TYPE, PCRecData.class);
			//typeMap.put(ProdRecData.DEFAULT_SQL_TYPE, ProdRecData.class);
			
			//	Since the SP uses temporary tables, make sure the connection
			//	does not commit anything
			//
			con.setAutoCommit(false);
			//	Set up the call to the SP & execute it.
			//
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_RECOMMENDATION.getPCRecStats(?,?,?,?)}");  
			cs.setLong (1, user.id);
			cs.registerOutParameter(2, Types.STRUCT, UserEvalData.DEFAULT_SQL_TYPE);
			cs.registerOutParameter(3, OracleTypes.CURSOR );
			cs.registerOutParameter(4, Types.INTEGER );	
			cs.executeQuery();
			
			//	Check for errors caught by the SP.
			//
			int error = cs.getInt(4);
			if (error != 0) 
				throw new AlkExcept("", error);
					
			//	Retrieve the UserEvalData object
			//
			evalData = (UserEvalData)cs.getObject(2);
			
			//	Retrieve the PCRecData objects.
			//
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(3); 
			while (rs.next()) 	{
				//PCRecData data = (PCRecData)rs.getObject(1);
				PCRecData data = PCRecData.fromResultSet(rs);
				ProductCluster pc = new ProductCluster(data.pcID);
				retMap.put(pc, data);
			}
			rs.close();
			cs.close();
			return retMap;
			
		} //END try
		catch(AlkExcept ae) {
			throw ae;
		}
		catch(SQLException se) 	{
			throw new AlkExcept("SQL Exception",6004, se);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage() + " in " + mName, 2504);
		}
		finally {
			try { con.close();	}	catch(Exception e) {}
		}
	}
	
	/**
	 * Obtains the predicted rating for the given User and Product. Calls the Stored Procedure <b><tt>pkg_Alkindi_Recommendation.getPredictedRating</tt></b>.
	 * @param user the SystemUser for whom the predicted rating data
	 * @param prod the product for which the predicted rating data
	 * @return The predicted rating as a float.
	 * @throws AlkExcept
	 * @roseuid 3A4B6610004E
	 */
	public float getPredictedRating(SystemUser user, Product prod) throws AlkExcept 
	{
		Connection con = null;
		int predictedRating = 0 ;
		ArrayList retList = new ArrayList();
		float predEval = 0f;
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = 
				con.prepareCall("{ call pkg_ALKINDI_RECOMMENDATION.getPredictedRating(?,?,?,?,?)}");  
			cs.setLong (1, user.id);
			cs.setInt (2, prod.id);
			cs.setDouble(3, propAPrime);
			cs.registerOutParameter(4, Types.FLOAT);
			cs.registerOutParameter(5, Types.INTEGER );
			cs.executeQuery();
			//	Check CS for error & assign default PR in that case.
			//
			int err = cs.getInt(5);
			if (err == 0) 
				predEval = cs.getFloat(4);			
			else
				predEval = propDefaultPR;				
			cs.close();
			
		} //END try

		catch(SQLException se) 		{
			throw new AlkExcept(se.getMessage(),6004);
		} 
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2506);
		}
		finally {
			try { con.close();	}	catch(Exception e) {}
		}
		return predEval;
	}
	
	/**
	 * Retrieves data for the Primary Products in the given ProductCluster for the given User. Uses the stored procedure <tt>pkg_ALKIND_RECOMMENDATION.getPrimaryProdRecsForUser</tt>.
	 * @param user The user in question
	 * @param pcID Identifies the ProductCluster from which to recommend
	 * @return An ArrayList of ProdRecData objects, each representing a candidate for recommending.
	 * @param numToGet The maximum number of products for which to return data.
	 * @throws AlkExcept
	 * @roseuid 3A9EA3FD0148
	 */
	public java.util.ArrayList getPrimaryProductData(final Alkindi.Data.SystemUser user, final int pcID, final int numToGet) throws AlkExcept 
	{
		final String mName = "getPrimaryProductData";
//		LogManager.dbgLog(cName, mName, "uid=" + user.id + " pcid = " + pcid + " ScoringFcn = " + ScoringFunction);
		Connection con = null;
		//	Create return object
		//
		ArrayList retList = new ArrayList();

		try
		{
			con = AlkConn.getConnection();
			
         	//	Map database object types to Java objects for this connection.
			//
			Map typeMap = con.getTypeMap();
			typeMap.put(UserEvalData.DEFAULT_SQL_TYPE, UserEvalData.class);
			
			//	Temporarily remove other object types from connection map
			//
			//typeMap.put(PCRecData.DEFAULT_SQL_TYPE, PCRecData.class);
			//typeMap.put(ProdRecData.DEFAULT_SQL_TYPE, ProdRecData.class);
			
			//	Since the SP uses temporary tables, make sure the connection
			//	does not commit anything
			//
			con.setAutoCommit(false);
			
			//	Prepare the call to the stored proc & execute.
			//
			CallableStatement cs = con.prepareCall("{ call  pkg_ALKINDI_RECOMMENDATION.GETPRIMARYRECPRODSFORUSER(?,?,?,?,?)}");  

			cs.setLong (1, user.id);
			cs.setInt (2, pcID);
			cs.setInt(3, numToGet);
			cs.registerOutParameter(4, OracleTypes.CURSOR );
			cs.registerOutParameter(5, Types.INTEGER );	
			cs.execute();
			
			//	Check for errors caught by the stored proc.
			//
			int error = cs.getInt(5);
			if (error != 0)
				throw new AlkExcept("DB Error " + error + " in " + mName, 6012);

			//	Walk the cursor of returned product data, parsing each item into 
			//	a ProdRecData object.
			//
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(4); 
			while (rs.next()){
				//ProdRecData data = (ProdRecData)rs.getObject(1);
				ProdRecData data = ProdRecData.fromResultSet(rs, propNumSF);
				retList.add(data);
			} //while
			rs.close();
			cs.close();
		} 
		catch(AlkExcept ae) {
			throw ae;
		}
		catch(SQLException se) 
		{
			//LogManager.err(cName, mName, se.getMessage());
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
		}
		catch(Exception e) {
			//LogManager.err(cName, mName, e.getMessage());
			throw new AlkExcept("Unexpected error in " + mName + ": " + e.getMessage(), 2512);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
		return retList;
	}
	
	/**
	 * 	Returns an ArrayList of Double objects of the normalized number of products in each Product Cluster.
	 * 	@return ArrayList
	 * 	@throws AlkExcept
	 * @roseuid 3A4B661000BB
	 */
	public ArrayList getProductCountByPC() throws AlkExcept 
	{
		final String mName = "getProductCountByPC";
		ArrayList counts = new ArrayList();
		Connection con = null;
		try {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_RECOMMENDATION.SP_SEL_PC_NI(?,?)}");
			cs.registerOutParameter(1, OracleTypes.CURSOR);
			cs.registerOutParameter(2, Types.INTEGER);
			cs.execute();
			int error = cs.getInt(2);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(1); 
			
			while(rs.next()) {
				counts.add(new Double(rs.getLong("NI")));
			}

			rs.close();
			cs.close();
		
		}
		catch (SQLException E) {
			String errMsg = E.toString();
			//LogManager.err(cName, mName, errMsg);
			throw new AlkExcept("SQL Exception in getProductCountByPC: " + errMsg, 2501);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2511);
		}
		finally {
			try { con.close(); } catch (Exception e) {}
		}
		return counts;
	}
	
	/**
	 * Retrieves data for the non-excluded Products in the given ProductCluster for the given User. Uses the stored procedure <tt>pkg_ALKIND_RECOMMENDATION.getProdRecsForUser</tt>.
	 * @param user The user in question
	 * @param pcID Identifies the ProductCluster from which to recommend
	 * @param lowRatingThreshold If the User rated a Product, the evaluation below which the Product is excluded from the returned list.
	 * @param recentRecThreshold The minimum number of days that a Product can have been previously recommended to this User. 
	 * @param numToGet The maximum number of products for which to return data.
	 * @return An ArrayList of ProdRecData objects, each representing a candidate for recommending.
	 * @throws AlkExcept
	 * @roseuid 3A4B661000BC
	 */
	public java.util.ArrayList getProductData(SystemUser user, int pcID, double lowRatingThreshold, final int recentRecThreshold, final int numToGet) throws AlkExcept 
	{
		final String mName = "getProductData";
//		LogManager.dbgLog(cName, mName, "uid=" + user.id + " pcid = " + pcid + " ScoringFcn = " + ScoringFunction);
		Connection con = null;
		//	Create return object
		//
		ArrayList retList = new ArrayList();

		try
		{
			con = AlkConn.getConnection();
			
			
         	//	Map database object types to Java objects for this connection.
			//
			Map typeMap = con.getTypeMap();
			typeMap.put(UserEvalData.DEFAULT_SQL_TYPE, UserEvalData.class);
			
			//	Temporarily remove other object types from connection map
			//
			//typeMap.put(PCRecData.DEFAULT_SQL_TYPE, PCRecData.class);
			//typeMap.put(ProdRecData.DEFAULT_SQL_TYPE, ProdRecData.class);
			
			//	Since the SP uses temporary tables, make sure the connection
			//	does not commit anything
			//
			con.setAutoCommit(false);
			
			//	Prepare the call to the stored proc & execute.
			//
			CallableStatement cs = con.prepareCall("{ call  pkg_ALKINDI_RECOMMENDATION.GETRECPRODSFORUSER(?,?,?,?,?,?,?)}");  

			cs.setLong (1, user.id);
			cs.setInt (2, pcID);
			cs.setDouble(3, lowRatingThreshold);
			cs.setInt (4, recentRecThreshold);
			cs.setInt(5, numToGet);
			cs.registerOutParameter(6, OracleTypes.CURSOR );
			cs.registerOutParameter(7, Types.INTEGER );	
			cs.execute();
			
			//	Check for errors caught by the stored proc.
			//
			int error = cs.getInt(7);
			if (error != 0)
				throw new AlkExcept("DB Error " + error + " in " + mName, 6012);

			//	Walk the cursor of returned product data, parsing each item into 
			//	a ProdRecData object.
			//
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(6); 
			while (rs.next()){
				//ProdRecData data = (ProdRecData)rs.getObject(1);
				ProdRecData data = ProdRecData.fromResultSet(rs, propNumSF);
				retList.add(data);
			} //while
			rs.close();
			cs.close();
		} 
		catch(AlkExcept ae) {
			throw ae;
		}
		catch(SQLException se) 
		{
			//LogManager.err(cName, mName, se.getMessage());
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
		}
		catch(Exception e) {
			//LogManager.err(cName, mName, e.getMessage());
			throw new AlkExcept("Unexpected error in " + mName + ": " + e.getMessage(), 2512);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
		return retList;
	}
	
	/**
	 * 	Retrieves an array of double values which can be used to select a Scoring Function.
	 * 	@return double[]
	 * 	@throws AlkExcept
	 * @roseuid 3A4B6610003F
	 */
	public double[] getScoreWeight() throws AlkExcept 
	{
		Connection con = null;
	    double [] dl = new double[propNumSF];
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = 
				con.prepareCall("{ call pkg_ALKINDI_RECOMMENDATION.sp_SEL_SCORING_FUNCTION_WEIGHT(?,?)}");  
			cs.registerOutParameter(1, OracleTypes.CURSOR );
			cs.registerOutParameter(2, Types.INTEGER );	
			cs.executeQuery();
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(1); 
			int error = cs.getInt(2);
			int i=0;
			if (error != 0) {
					throw new AlkExcept("", error);
			}
			while (rs.next()) 
			{
			dl[i] = rs.getDouble ("SCORING_FUNCTION_WEIGHT"); 
			i++;
			}
			rs.close();
	       	cs.close();
		} //END try

		catch(SQLException se) 
		{
			throw new AlkExcept("SQL Exception in RecMgrDBUtils.getScoreWeight.",6004);
		} 
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2505);
		}
		finally {
			try {	con.close(); }	catch(Exception e) {	}
		}
	return dl;
	}
	
	/**
	 * 	Inserts the given recommendations into the database.
	 * 	@param recsfList an ArrayList of RecWithScoringFcn objects.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B6610007D
	 */
	public void insertRecommendations(RecommendationList recsToInsert) throws AlkExcept 
	{
		Connection con = null;
		try {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_RECOMMENDATION.sp_INS_Recs(?,?,?,?,?) }");
			Iterator recsIt = recsToInsert.iterator();
			while (recsIt.hasNext()) {
				Recommendation rec = (Recommendation)recsIt.next();
				cs.setLong(1, rec.user.id);
				cs.setLong(2, rec.prod.id);
				cs.setDouble(3, rec.predictedRating());
				cs.setInt(4, rec.scoreFuncID);
				cs.setInt(5, rec.prodClusterID);
				cs.executeUpdate();
			}
           	cs.close();

       	} 
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in RecMgrDBUtils.insertRecommendations. Message: " 
				+ se.getMessage(),6004);
		}	
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2508);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * Constructor to read Engine properties.
	 * @roseuid 3A4B6610002E
	 */
	public RecMgrDBUtils() throws AlkExcept 
	{
		propNumSF = PropertyManager.getInt("NumSF");
		propB6 = PropertyManager.getInt("B6");
		propAPrime = PropertyManager.getDouble("APrime");
		propDefaultPR = PropertyManager.getFloat("DefaultPR");
	}
}
